#!/bin/bash
SCRIPT_PATH=`cd $(dirname $0);pwd -P`
# echo "SCRIPT_PATH: ${SCRIPT_PATH}"

# !!! Note: !!!
# Edit and Comment in English, please

# set the runtime-system in English
export LANG=""
export LANGUAGE=""

# stop linux sfesp func
EDR_SFESP=/etc/cron.d/edr_sfesp

# under HCI proxy, the mgr's ip address and download port
G2HPorxyIP=127.0.0.1
G2HProxyPort=18524
G2HProxyName=G2HProxy
NET_TOOL=netstat

ERR_NET_CONNECT=1026
ERR_MEMORY_SHORTAGE=1054

#Different error codes from Windows Terminal (1010-1024)
ERR_ALREADY_INSTALL=1010
ERR_ROOT_PRIVILIGE=1011
ERR_INVALID_PARAM=1012
ERR_COMMAND_MISS=1013
ERR_INSTALL_POSITION=1014
ERR_INSTALL_PATH=1015
ERR_CRON_SERVICE=1016
ERR_G2H_PORT_OCCUPIED=1017
ERR_UNSUPPORT_SYSTEM=1018
ERR_INSTALL_CHECK=1019

# system support list
WHITELIST=("CentOS 5 x86" "CentOS 5 x64" "CentOS 6 x86" "CentOS 6 x64" "CentOS 7 x64" 
            "Ubuntu 10 x86" "Ubuntu 10 x64" "Ubuntu 11 x86" "Ubuntu 11 x64" "Ubuntu 12 x86" 
            "Ubuntu 12 x64" "Ubuntu 13 x86" "Ubuntu 13 x64" "Ubuntu 14 x86" "Ubuntu 14 x64" 
            "Ubuntu 16 x86" "Ubuntu 16 x64" "Ubuntu 17 x86" "Ubuntu 17 x64" "Ubuntu 18 x86" 
            "Ubuntu 18 x64" "Ubuntu 20 x86" "Ubuntu 20 x64" 
            "Debian 6 x86" "Debian 6 x64" "Debian 7 x86" "Debian 7 x64" "Debian 8 x86" 
            "Debian 8 x64" "Debian 9 x86" "Debian 9 x64" 
            "RHEL 5 x86" "RHEL 5 x64" "RHEL 6 x86" "RHEL 6 x64" "RHEL 7 x64" 
            "SUSE 11" "SUSE 12" "SUSE 15" 
            "Oracle 5 x86" "Oracle 5 x64" "Oracle 6 x86" "Oracle 6 x64" "Oracle 7 x64" "AlmaLinux x64"
)

OS_NAME=
OS_VERSION=
OS_ARCHITECTURE=
OS_PLATFORMID=
OS_PLATFORM=
VERSION_WILDCARD=x

# Minimum Hardware Specifications
MEMORY_REQUIREMENT_GB=4
CPU_REQUIREMENT_CORES=2

# recomend Hardware Specifications
MEMORY_RECOMMENDED_GB=4
CPU_RECOMMENDED_CORES=4

# release os limit。Elements in xxx_Support_List without architecture, such as "16", represent the minimum version number of the gray system. This is because there is a gray system between the red line and whitelist for some releases in the compatibility list.
CentOS_Support_List=("7 x64" "7.1 x64" "7.2 x64" "7.3 x64" "7.4 x64" "7.5 x64" "7.6 x64" "7.7 x64" "7.8 x64" "7.9 x64" 
                     "8 x64" "8.1 x64" "8.2 x64" "8.3 x64" "8.4 x64" "8.5 x64"
)

Ubuntu_Support_List=("18.04 x64" "20.04 x64" "22.04 x64" "24.04 x64"
)

Debian_Support_List=("9.1 x64" "9.1 x86" "9.6 x64" "9.6 x86" "9.7 x64" "9.7 x86" "9.13 x64" "9.13 x86" 
                      "11.9 x64" "11.9 x86" "12.5 x64" "12.5 x86"
)

RHEL_Support_List=("7.1 x64" "7.2 x64" "7.3 x64" "7.4 x64" "7.5 x64" "7.6 x64" "7.7 x64" "7.8 x64" "7.9 x64" 
                   "8 x64" "8.2 x64" "8.4 x64" "8.5 x64" "8.6 x64" "8.7 x64" "8.8 x64" "8.9 x64" 
                   "9.0 x64" "9.1 x64" "9.2 x64" "9.3 x64" "9.4 x64"
)

SUSE_Support_List=("12" "15.$VERSION_WILDCARD x64"
)

Oracle_Support_List=("7" "8.$VERSION_WILDCARD x64" "9.$VERSION_WILDCARD x64"
)

Rocky_Support_List=("8.4 x64" "8.7 x64" "8.8 x64" "8.10 x64" "9.2 x64"
)

AlmaLinux_Support_List=("8.3 x64" "8.9 x64" "8.10 x64" "9.2 x64" "9.3 x64" "9.4 x64" "9.5 x64"
)

ALinux_Support_List=("2.1903 x64" "3.2104 x64"
)

memTotalGB=0
cpuCores=0

# get os name
obtainOpSystem(){
    while true; do
        # through lsb_releaseget release version 
        [ $(which lsb_release 2>/dev/null) ] && OS_NAME="$(lsb_release -d |grep -iEo "CentOS|Ubuntu|Debian|Red|SUSE|Oracle|AlmaLinux|Rocky|Alibaba Cloud Linux"|tr a-z A-Z)"
        if [[ -n "$OS_NAME" ]]; then
            break
        fi

        # through /etc/issue get release version
        [ -f /etc/issue ] && OS_NAME="$(cat /etc/issue |grep -iEo "CentOS|Ubuntu|Debian|Red|SUSE|Oracle|AlmaLinux|Rocky|Alibaba Cloud Linux"|tr a-z A-Z)"
        if [[ -n "$OS_NAME" ]]; then
            break
        fi

        # through /etc/os-release get release version
        [ -f /etc/os-release ] && OS_NAME="$(cat /etc/os-release|grep -E "^(PRETTY_NAME)="|grep -iEo "CentOS|Ubuntu|Debian|Red|SUSE|Oracle|AlmaLinux|Rocky|Alibaba Cloud Linux"|tr a-z A-Z)"
        if [[ -n "$OS_NAME" ]]; then
            break
        fi

        # through/etc/*release or /etc/*version check release version
        if [[ -f "/etc/oracle-release" ]]; then
            OS_NAME="ORACLE"
        elif [[ -f "/etc/almalinux-release" ]]; then
            OS_NAME="ALMALINUX"
        elif [[ -f "/etc/SuSE-release" ]]; then
            OS_NAME="SUSE"
        elif [[ -f "/etc/alinux-release" ]]; then
            OS_NAME="ALIBABA CLOUD LINUX"
        elif [[ -f "/etc/centos-release" ]]; then
            OS_NAME="CENTOS"
        elif [[ -f "/etc/redhat-release" ]]; then
            OS_NAME="RED"
        elif [[ -f "/etc/debian_version" ]]; then
            OS_NAME="DEBIAN"
        elif [[ -f "/etc/rocky-release" ]]; then
            OS_NAME="ROCKY"
        fi
        break
    done

    case $OS_NAME in
        "CENTOS")
            OS_NAME=CentOS;;
        "UBUNTU")
            OS_NAME=Ubuntu;;
        "DEBIAN")
            OS_NAME=Debian;;
        "RED")
            OS_NAME=RHEL;;
        "SUSE")
            OS_NAME=SUSE;;
        "ORACLE")
            OS_NAME=Oracle;;
        "ALMALINUX")
            OS_NAME=AlmaLinux;;
        "ROCKY")
            OS_NAME=Rocky;;
        "ALIBABA CLOUD LINUX")
            OS_NAME=ALinux;;
    esac
}

# get os-release version
obtainOpSystemVersion(){
    local detail_os_version=""
    local os_id=""
    # obtain the version information through lsb_release
    [ $(which lsb_release 2>/dev/null) ] && detail_os_version=$(lsb_release -r | awk -F "\t" '{print $2}')
    if [[ ${#detail_os_version} -gt ${#OS_VERSION} ]]; then
        OS_VERSION=$detail_os_version
    fi

    # obtain the version information through /etc/os-release
    if [ -f /etc/os-release ]; then
      detail_os_version=$(grep -E "^(VERSION_ID)=" /etc/os-release | awk -F "=" '{print $NF}' | tr -d '[:space:]"')
      if [[ ${#detail_os_version} -gt ${#OS_VERSION} ]]; then
          OS_VERSION=$detail_os_version
      fi
      os_id=$(grep -E "^(ID)=" /etc/os-release | awk -F "=" '{print $NF}' | tr -d '[:space:]"' | tr '[:upper:]' '[:lower:]')
    fi

    # Red Hat-based distributions obtain the version information through /etc/redhat-release
    [ -f /etc/redhat-release ] && detail_os_version=$(cat /etc/redhat-release | sed -n 's/.*release \([0-9.]*\).*/\1/p')
    if [[ ${#detail_os_version} -gt ${#OS_VERSION} ]]; then
        OS_VERSION=$detail_os_version
    fi

    # obtain the version information through /etc/debian_version
    [ -f /etc/debian_version ] && detail_os_version=$(cat /etc/debian_version|tr -cd [0-9.])
    if [[ ${#detail_os_version} -gt ${#OS_VERSION} ]]; then
        OS_VERSION=$detail_os_version
    fi

    # Extract two segments of the version number, such as 7.9; if the version number has only one segment, extract just that one, such as 7.
    IFS='.' read -r major minor rest <<< "$OS_VERSION"
    
    if [[ "$os_id" == "alinux" ]]; then
        local alinux_minor_id=""
        alinux_minor_id=$(grep -E "^(ALINUX_MINOR_ID)=" /etc/os-release | awk -F "=" '{print $NF}' | tr -d '[:space:]"')
        if [[ -n "$alinux_minor_id" ]]; then
            minor="$alinux_minor_id"
        fi
    fi
    OS_VERSION=${major}${minor:+.$minor}
}

# get os runtime bit
obtainSystemBit(){
    OS_ARCHITECTURE=$(getconf LONG_BIT)
    case $OS_ARCHITECTURE in 
    64)
        OS_ARCHITECTURE=x64;;
    32)
        OS_ARCHITECTURE=x86;;
    esac
}

# check if the OS is in 'system support list'
isInWhilteList(){
    local main_version=${OS_VERSION%%.*}
    if [[ "$OS_NAME" == "SUSE" ]]; then
        currSystem="$OS_NAME $main_version"
    else
        currSystem="$OS_NAME $main_version $OS_ARCHITECTURE"
    fi
    for(( i=0; i<${#WHITELIST[@]}; i++))
    do
        if [[ "$currSystem" == "${WHITELIST[i]}" ]]; then
            return 0
        fi        
    done
    echo "Warning:Current system:$currSystem, Not support,${WHITELIST[@]}"
    return 1
}

# main OS check precess
checkOpSystem(){
    obtainOpSystem
    if [[ -z "$OS_NAME" ]]; then
        echo "Failed to obtain operating system!"
        return 1
    fi
    obtainOpSystemVersion
    if [[ -z "$OS_VERSION" ]]; then
        echo "Failed to obtain the operating system version!"
        return 1
    fi
    obtainSystemBit
    if [[ -z "$OS_ARCHITECTURE" ]]; then
        echo "Failed to obtain system hardware architecture!"
        return 1
    fi
    isInWhilteList
    if [[ $? -ne 0 ]]; then
        return 1
    fi
    return 0
}

# can't not be installed in domestic OS.
# if returnVal == 1, should stop installing now.
checkIsGchOpSystem() {
    OS_PLATFORMID=`cat /etc/os-release | sed 's/^[ \t]*//g' | grep ^ID= | sed 's/ID=\([0-9a-zA-Z-]*\)/\1/g' | awk '{print tolower($1)}'`
    OS_PLATFORM=$(uname -m)
    OS_SYSBIT=$(getconf LONG_BIT)
    
    if [ "${OS_SYSBIT}" != "64" ]; then
		return 1
    fi

    if [[ "${OS_PLATFORM}" != *"x86_64"* ]] && [[ "${OS_PLATFORM}" != *"i386"* ]] && [[ "${OS_PLATFORM}" != *"i686"* ]]; then
        # echo "This operate system id is ${OS_PLATFORMID}, bit is ${OS_SYSBIT}, this is domestic operate system"
        return 0
    fi

    if [ -z "$OS_PLATFORMID" ]; then
        return 1
    fi

    if [[ "${OS_PLATFORMID}" = *"kylin"* ]] || \
       [[ "${OS_PLATFORMID}" = *"uos"* ]] || [[ "${OS_PLATFORMID}" = *"deepin"* ]] || \
       [[ "${OS_PLATFORMID}" = *"nfs"* ]] || [[ "${OS_PLATFORMID}" = *"openeuler"* ]] || \
       [[ "${OS_PLATFORMID}" = *"euleros"* ]] || [[ "${OS_PLATFORMID}" = *"asianux"* ]] || \
       [[ "${OS_PLATFORMID}" = *"redflag"* ]] || [[ "${OS_PLATFORMID}" = *"linx"* ]] || \
       [[ "${OS_PLATFORMID}" = *"anolis"* ]]; then
            # echo "This operate system id is ${OS_PLATFORMID}, this is domestic operate system"
            return 0
    fi

	return 1
}

function checkip() { 
	regex="\b(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[1-9])\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[0-9])\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[0-9])\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[0-9])\b"
	ckStep2=`echo $1 | egrep $regex | wc -l`
	if [ $ckStep2 -eq 0 ]
	then
       return 1
	else
       return 0
	fi
}

#check regex,match return 1, no match return 0
function checkegrep()
{
    dest=$1
    regex_rule=$2
    ckStep2=`echo $dest | egrep $regex_rule 2>/dev/null | wc -l`
    if [ $ckStep2 -eq 0 ]
    then
        return 0
    else
        return 1
    fi
}

#check ip,dns,ipv6,match return 1，no match return 0
checkdest()
{
    #checkip
    regex_ip="\b(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[1-9])\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[0-9])\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[0-9])\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[0-9])\b"
    checkegrep $1 $regex_ip
    if [ $? -eq 1 ];then
        return 1
    fi
    #checkDns
    regex_dns="^[a-zA-Z]{1}[-a-zA-Z0-9]{0,61}(\.[a-zA-Z0-9][-a-zA-Z0-9]{0,62}){2,3}$"
    checkegrep $1 $regex_dns
    if [ $? -eq 1 ];then
        return 1
    fi
    #check ipv6
    regex_ipv6_1="^((([0-9A-Fa-f]{1,4}:){7}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){1,7}:)|(([0-9A-Fa-f]{1,4}:){6}:[0-9A-Fa-f]{1,4})$"
    regex_ipv6_2="^(([0-9A-Fa-f]{1,4}:){5}(:[0-9A-Fa-f]{1,4}){1,2})|(([0-9A-Fa-f]{1,4}:){4}(:[0-9A-Fa-f]{1,4}){1,3})$"
    regex_ipv6_3="^(([0-9A-Fa-f]{1,4}:){3}(:[0-9A-Fa-f]{1,4}){1,4})|(([0-9A-Fa-f]{1,4}:){2}(:[0-9A-Fa-f]{1,4}){1,5})$"
    regex_ipv6_4="^([0-9A-Fa-f]{1,4}:(:[0-9A-Fa-f]{1,4}){1,6})|(:(:[0-9A-Fa-f]{1,4}){1,7})$"
    regex_ipv6_5="^(([0-9A-Fa-f]{1,4}:){6}(\\d|[1-9]\\d|1\\d{2}|2[0-4]\\d|25[0-5])(\\.(\\d|[1-9]\\d|1\\d{2}|2[0-4]\\d|25[0-5])){3})$"
    regex_ipv6_6="^(([0-9A-Fa-f]{1,4}:){5}:(\\d|[1-9]\\d|1\\d{2}|2[0-4]\\d|25[0-5])(\\.(\\d|[1-9]\\d|1\\d{2}|2[0-4]\\d|25[0-5])){3})$"
    regex_ipv6_7="^(([0-9A-Fa-f]{1,4}:){4}(:[0-9A-Fa-f]{1,4}){0,1}:(\\d|[1-9]\\d|1\\d{2}|2[0-4]\\d|25[0-5])(\\.(\\d|[1-9]\\d|1\\d{2}$"
    regex_ipv6_8="^2[0-4]\\d|25[0-5])){3})|(([0-9A-Fa-f]{1,4}:){3}(:[0-9A-Fa-f]{1,4}){0,2}:(\\d|[1-9]\\d|1\\d{2}$"
    regex_ipv6_9="^2[0-4]\\d|25[0-5])(\\.(\\d|[1-9]\\d|1\\d{2}|2[0-4]\\d|25[0-5])){3})$"
    regex_ipv6_10="^(([0-9A-Fa-f]{1,4}:){2}(:[0-9A-Fa-f]{1,4}){0,3}:(\\d|[1-9]\\d|1\\d{2}|2[0-4]\\d|25[0-5])(\\.(\\d|[1-9]\\d|1\\d{2}|2[0-4]\\d|25[0-5])){3})$"
    regex_ipv6_11="^([0-9A-Fa-f]{1,4}:(:[0-9A-Fa-f]{1,4}){0,4}:(\\d|[1-9]\\d|1\\d{2}$"
    regex_ipv6_12="^2[0-4]\\d|25[0-5])(\\.(\\d|[1-9]\\d|1\\d{2}|2[0-4]\\d|25[0-5])){3})$"
    regex_ipv6_13="^(:(:[0-9A-Fa-f]{1,4}){0,5}:(((\\d{1,2})|(1\\d{2})|(2[0-4]\\d)|(25[0-5]))\\.){3}((\\d{1,2})|(1\\d{2})|(2[0-4]\\d)|(25[0-5]))))$"
    checkegrep $1 $regex_ipv6_1
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_2
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_3
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_4
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_5
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_6
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_7
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_8
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_9
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_10
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_11
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_12
    if [ $? -eq 1 ];then
        return 1
    fi
    checkegrep $1 $regex_ipv6_13
    if [ $? -eq 1 ];then
        return 1
    fi
    return 0
}


function checkport() {
	if [ -z $1 ];then
		return 1
	fi
	
	if [ $1 -gt 65535 ] || [ $1 -lt 1 ] ;then
		return 1
	fi
	return 0
}

compare_versions() {
    local version1=(${1//./ })
    local version2=(${2//./ })
    local max_length=${#version1[@]}

    # Ensure that two version number arrays have the same length.
    if [ ${#version2[@]} -gt $max_length ]; then
        max_length=${#version2[@]}
    fi

    # Compare each part of the version number individually
    for ((i=0; i<max_length; i++)); do
        # if not exist, set 0
        [ -z ${version1[i]} ] && version1[i]=0
        [ -z ${version2[i]} ] && version2[i]=0

        #  Compare the numerical size of each part
        if [[ ${version1[i]} == ${VERSION_WILDCARD} ]] || [[ ${version2[i]} == ${VERSION_WILDCARD} ]];then
            # If a version number wildcard is encountered, it is considered the same.
            echo "0"
            return
        elif ((10#${version1[i]} > 10#${version2[i]})); then
            echo "1"
            return
        elif ((10#${version1[i]} < 10#${version2[i]})); then
            echo "-1"
            return
        fi
    done

    echo "0"
}

# 0 whitelist；1 grey sys；2 redline sys
checkOpSystemVersion(){
    if [[ -z "$OS_NAME" ]]; then
        obtainOpSystem
        if [[ -z "$OS_NAME" ]]; then
            echo "Failed to obtain operating system!"
            return 1
        fi
    fi

    if [[ -z "$OS_VERSION" ]]; then
        obtainOpSystemVersion
        if [[ -z "$OS_VERSION" ]]; then
            echo "Failed to obtain the operating system version!"
            return 1
        fi
    fi

    if [[ -z "$OS_ARCHITECTURE" ]]; then
        obtainSystemBit
        if [[ -z "$OS_ARCHITECTURE" ]]; then
            echo "Failed to obtain the operating system architecture!"
            return 1
        fi
    fi

    local index=0
    local composed_var_name="${OS_NAME}_Support_List[@]"
    eval "local supported_versions=(\"\${$composed_var_name}\")"
    for element in "${supported_versions[@]}"; do
        IFS=' ' read -ra ADDR <<< "${element}"
        local version=${ADDR[0]}
        local architecture=${ADDR[1]}
        local result=$(compare_versions ${version} ${OS_VERSION})
        if [[ "$result" -eq "1" ]]; then
            if [[ "$index" -eq "0" && "$OS_NAME" != "AlmaLinux" && "$OS_NAME" != "Rocky" ]]; then
              
                return 2
            fi
        elif [[ "$result" -eq "0" ]]; then
            if [[ ${architecture} == ${OS_ARCHITECTURE} ]]; then
                return 0
            fi
        fi
        ((index++))
    done

    return 1
}

# 。0 whitelist；1 grey sys；2 redline sys
checkHardware(){
    # To retrieve memory capacity, convert it to GB, and round to one decimal place.
    which dmidecode >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        #  get BIOS via dmidecode
        local memTotalMB=0
        local memory_info=$(dmidecode --type memory 2>/dev/null)
        while read -r line; do
            if [[ $line == *"Size:"* && $line != *"No Module Installed"* ]]; then
                # get num adn unit
                local size=$(echo $line | sed -n 's/.*Size: \+\([0-9]\+\) \([MG]\)B.*/\1/p')
                local unit=$(echo $line | awk '/Size:/ {print $3}')

                # Based on the unit, convert to MB and add to the total memory.
                if [ "$unit" == "GB" ]; then
                    memTotalMB=$((memTotalMB + size * 1024))
                elif [ "$unit" == "MB" ]; then
                    memTotalMB=$((memTotalMB + size))
                fi
            fi
        done <<< "$memory_info"
        local GBToMB=1024
        if [ $((memTotalMB % GBToMB)) -gt $((GBToMB / 2)) ]; then
            memTotalGB=$(((memTotalMB / GBToMB) + 1))
        else
            memTotalGB=$((memTotalMB / GBToMB))
        fi
        # echo "dmidecode memTotalGB:[$memTotalGB]"
    fi

    if [ $memTotalGB -eq 0 ]; then
        # read /proc/meminfo 
        local memTotalKB=$(awk '/MemTotal/{print $2}' /proc/meminfo)
        local GBToKB=$((1024 * 1024))
        if [ $((memTotalKB % GBToKB)) -gt $((GBToKB / 2)) ]; then
            memTotalGB=$(((memTotalKB / GBToKB) + 1))
        else
            memTotalGB=$((memTotalKB / GBToKB))
        fi
        # echo "meminfo memTotalGB:[$memTotalGB]"
    fi

    # get cpu
    cpuCores=$(grep -c processor /proc/cpuinfo)
    # echo "cpuCores:[$cpuCores]"

    if [[ "$memTotalGB" -lt $MEMORY_REQUIREMENT_GB ]] || [[ "$cpuCores" -lt $CPU_REQUIREMENT_CORES ]];then
        return 2
    elif [[ "$memTotalGB" -ge $MEMORY_RECOMMENDED_GB ]] && [[ "$cpuCores" -ge $CPU_RECOMMENDED_CORES ]];then
        return 0
    else
        return 1
    fi
}

need_tool=()
cnt=0
function check_tool(){
    for i in $*;
	do
        which $i >/dev/null 2>&1
        if [ $? -ne 0 ]; then
	        need_tool[cnt]="$i"
		    cnt=`expr $cnt + 1`
	    fi
	done
}

# help
# -c for one-key deploy. the ipset interactive prompts will be ignored and installation packages will be automatically deleted.
# -e installing with HCI proxy
# -o full package installation
# -s check if the current OS is the one in 'support list'
# -u saseEdr enterprise id(in sase case)
return_usage() {
	echo -e "Usage: $0 [OPTION]..."
	echo -e "Example [1]: $0 \\033[40;31m(Use when there is manager_info.txt in the same directory)\\033[0m"
	echo -e "        [2]: $0 -h 8.8.8.8 -p 443 -f"
	echo -e "  -h host       set mgr host"
	echo -e "  -p port       set mgr port"
	echo -e "  -d dir        set absolute install dir"
	echo -e "  -c            slient install"
	echo -e "  -e            proxy install"
	echo -e "  -o            full package offline install"
	echo -e "  -s            system check"
	echo -e "  -u            saseEdr enterprise id"
	echo -e "  --help        display this help"
	exit $ERR_INVALID_PARAM
}

slient_install=0
force_install=0
proxy_install=0
system_check=0
full_package_install=0
install_check=1
force_install_allowed=0
mgr_ips=
enable_cron_cmd=

SHORT=h:p:d:u:g:fceso
LONG=ignore-install-check,force-install-allowed
OPTS=$(getopt -o $SHORT --long $LONG --name "$0" -- "$@")
if [ $? != 0 ]; then
	return_usage
fi
eval set -- "$OPTS"


while true; do
	case "$1" in
		-h)
			avai_ip=$2
			shift 2
			;;
		-p)
			avai_port=$2
			shift 2
			;;
		-d)
			installdir=$2
			shift 2
			;;
		-f)
			force_install=1
			shift
			;;
		-c)
			slient_install=1
			shift
			;;
		-o)
			full_package_install=1
			shift
			;;
		-e)
			proxy_install=1
			shift
			;;
		-s)
			system_check=1
			shift
			;;
		-u)
			szuid=$2
			shift 2
			;;
		--ignore-install-check)
			install_check=0
			shift
			;;
		--force-install-allowed)
			force_install_allowed=1
			shift
			;;
		-g)
			zonename=$2
			shift 2
			;;
		--)
			shift
			break
			;;
		*)
			return_usage
			;;
	esac
done

function ReadINIfile()
{     
    Key=$1  
    Section=$2
    Configfile=$3
    ReadINI=`awk -F '=' '/\['$Section'\]/{a=1}a==1&&$1~/'$Key'/{print $2;exit}' $Configfile | tr -d " "`
    echo "$ReadINI"
}

curr_dir=$(pwd)
ips_info=$curr_dir"/manager_info.txt"

uname -a | grep "x86_64" > /dev/null
if [ $? -ne 0 ] ; then
	# echo "edr agent is installing on x86 machines"
	full_package=$curr_dir"/packages_86.tar.gz"
else
	# echo "edr agent is installing on x86_64 machines"
	full_package=$curr_dir"/packages_64.tar.gz"
fi

function get_port_info()
{
	# allows use of the entered tenant id
	# TD2023062000553: the full package is based on the input parameters and no longer reads files
	if [ "${szuid}" == "" ] && [ $full_package_install -ne 1 ];then
		szuid=`ReadINIfile "tenant_id" "config" "$ips_info"`
	fi

	if [ -z "$avai_port" ];then
		if [ -f $ips_info ]; then
			avai_port=`ReadINIfile "agt_download_port" "config" "$ips_info"`
		fi

		# manager.info is not exist in full package
		# docker env, default: 443
		# other env, default: 4430
		if [ -z $avai_port ];then
			if [ "${szuid}" == "" ];then
				avai_port=4430
			else 
				avai_port=443
			fi
		fi
	fi

	return 0
}

function check_addr_available()
{
	if [ -z "$1" ] || [ -z "$2" ]; then
		# echo "Error: IP address and port must be provided."
		return 1
	fi

	which bash >/dev/null 2>&1
	if [ $? -eq 0 ]; then
		which timeout >/dev/null 2>&1
		if [ $? -eq 0 ]; then
			if timeout 5 bash -c "cat < /dev/null > /dev/tcp/$1/$2" >/dev/null 2>&1; then
				# echo "Success: $1:$2 is reachable."
				return 0
			else
				# echo "Error: $1:$2 is not reachable."
				return 1
			fi
		else
			echo "timeout is not available."
			if bash -c "cat < /dev/null > /dev/tcp/$1/$2" >/dev/null 2>&1; then
				# echo "Success: $1:$2 is reachable."
				return 0
			else
				# echo "Error: $1:$2 is not reachable."
				return 1
			fi
		fi
	else
		if ping -c 3 "$1" >/dev/null 2>&1; then
			# echo "Success: $1 is reachable."
			return 0
		else
			# echo "Error: $1 is not reachable."
			return 1
		fi
	fi
}

function get_available_mgr_addr()
{
	bSucc=0
	if [ -z "$avai_ip" ];then
		if [ -f $ips_info ]; then
			addrCount=`ReadINIfile "count" "config" "$ips_info"`
			for((i=0; i<addrCount; i++))
			do
				Key="addr""$i"
				addr=`ReadINIfile "$Key" "config" "$ips_info"`
							checkdest $addr
				if [ $? -eq 0 ];then
						#no mache ip or ipv6 or dns
					continue
				fi

				# perform connectivity check on the MGR
				check_addr_available $addr $avai_port
				if [ $? -ne 0 ]; then
					# echo "$addr can't be connected"
					if [ -z "$mgr_ips" ]; then
						mgr_ips="$addr"
					else
						mgr_ips="$mgr_ips,$addr"
					fi
					continue
				else
					avai_ip=$addr
					bSucc=1
					# echo "$avai_ip is available"
					break
				fi
			done
		fi
	else
		checkdest $avai_ip
		if [ $? -ne 0 ];then
			if [ $full_package_install -ne 1 ];then
				check_addr_available $avai_ip $avai_port
				if [ $? -eq 0 ]; then
					bSucc=1
				fi
			else
				bSucc=1
			fi
		fi
		mgr_ips=$avai_ip
	fi

	if [ $bSucc -ne 1 ]; then
		if [ -z "$mgr_ips" ]; then
			echo -e "[Failed] Unable to connect to the manager. Please check the connectivity of the port ($avai_port) between the agent and the manager."
		else
			echo -e "[Failed] Unable to connect to the manager. Please check the connectivity to the manager address ($mgr_ips) and port ($avai_port) between the agent and the manager."
		fi
		return 1
	fi
	return 0
}

function get_need_tool()
{
	# `ss` can be replaced if `netstat` is not exists
	which netstat >/dev/null 2>&1
	if [ $? -ne 0 ]; then
		NET_TOOL=ss
	fi

	# Note: this procedure must be called at the beginning, otherwise we cannot ensure that all the required tools are already present.
	# before installation, all dependent tools are checked uniformly, and users are prompted to install the missing tools uniformly. 
	# if you need to add a check dependency tool, add it here directly
	check_tool grep iptables ip6tables iptables-restore iptables-save ip6tables-save sed awk df openssl wget crontab tar ${NET_TOOL}
}

function check_need_tool()
{
	len=${#need_tool[@]}

	if [ $len -ne 0 ];then
			# echo "Installer asks tools bellow:" # echo "Installer asks tools as following:" 
		echo -e "[Failed] The following tools need to be installed: "
		for v in ${need_tool[@]};
		do
				# echo "$v" 
			echo -e "         $v"
		done
		return 1
	else
		return 0
	fi
}

# ============================== cron status check start ==============================
# return Systemd
#        Upstart
#        Sysvinit
function system_init_check() {
	if systemctl --version >/dev/null 2>&1; then
		echo "systemd"
	elif initctl version >/dev/null 2>&1; then
		echo "upstart"
	else
		echo "sysvinit"
	fi
}

function systemd_status_service() {
	local rtn=$(systemctl status "$1")
	if [[ $rtn =~ "$1"".service; disabled" ]]; then
		return 1
	fi
	return 0
}

# configure service startup via chkconfig
#   $1: service (usually cron / crond)
function chkconfig_status_service() {
	local rtn=$(chkconfig --list | grep "$1")
	if [[ $rtn =~ "2:off" || $rtn =~ "3:off" ||
		$rtn =~ "4:off" || $rtn =~ "5:off" ]]; then
		return 1
	fi
	return 0
}

# configure service startup via upstart(only cron.)
function upstart_status_service() {
	if [ ! -f "/etc/init/cron.conf" ]; then
		return 1
	fi
	return 0
}

# configure service startup via update-rc.d(only cron.)
function updatercd_status_service() {
	if [[ ! -f $(echo /etc/rc2.d/S[0-9][0-9]cron) ||
		! -f $(echo /etc/rc3.d/S[0-9][0-9]cron) ||
		! -f $(echo /etc/rc4.d/S[0-9][0-9]cron) ||
		! -f $(echo /etc/rc5.d/S[0-9][0-9]cron) ]]; then
		return 1
	fi
	return 0
}

# RedHat,CentOS,Oracle,NeoKylin, usually systemd or sysvinit
function redhat_status_cron() {
	# echo "redhat"
	local rtn=$(system_init_check)
	if [[ $rtn == "systemd" ]]; then
		enable_cron_cmd="systemctl enable crond"
		systemd_status_service crond
		return $?
	elif [[ $rtn == "upstart" ]]; then
		if initctl status crond &>/dev/null; then
		# if command exec success, it means system can start cron service
			upstart_status_service
			return $?
		else
			enable_cron_cmd="chkconfig crond on"
			chkconfig_status_service crond
			return $?
		fi
	else
		enable_cron_cmd="chkconfig crond on"
		chkconfig_status_service crond
		return $?
	fi
}

# Ubuntu,UbuntuKylin,Kylin
function ubuntu_status_cron() {
	# echo "ubuntu"
	local rtn=$(system_init_check)
	if [[ $rtn == "systemd" ]]; then
		enable_cron_cmd="systemctl enable cron"
		systemd_status_service cron
		return $?
	elif [[ $rtn == "upstart" ]]; then
		upstart_status_service
		return $?
	else
		return 1
	fi
}

# Debian
function debian_status_cron() {
	# echo "debian"
	local rtn=$(system_init_check)
	if [[ $rtn == "systemd" ]]; then
		enable_cron_cmd="systemctl enable cron"
		systemd_status_service cron
		return $?
	elif [[ $rtn == "sysvinit" ]]; then
		enable_cron_cmd="update-rc.d cron enable && update-rc.d cron defaults"
		updatercd_status_service
		return $?
	else
		return 1
	fi
}

# SUSE
function suse_status_cron() {
	# echo "suse"
	local rtn=$(system_init_check)
	if [[ $rtn == "systemd" ]]; then
		enable_cron_cmd="systemctl enable cron"
		systemd_status_service cron
		return $?
	elif [[ $rtn == "sysvinit" ]]; then
		enable_cron_cmd="chkconfig cron on"
		chkconfig_status_service cron
		return $?
	else
		return 1
	fi
}

function check_cron_status()
{
	local status=1
	# load $lsb
	lsb=$(lsb_release -a 2>&1)

	# load $PRETTY_NAME
	if [ -f /etc/os-release ]; then
		source /etc/os-release
	fi

	if [ -f /etc/redhat-release ]; then
		redhat_status_cron
		status=$?
	elif [[ $lsb =~ "Ubuntu" || $PRETTY_NAME =~ "Ubuntu" ]]; then
		ubuntu_status_cron
		status=$?
	elif [[ $lsb =~ "Debian" || $PRETTY_NAME =~ "Debian" ]]; then
		debian_status_cron
		status=$?
	elif [[ $lsb =~ "SUSE" || $PRETTY_NAME =~ "SUSE" ]]; then
		suse_status_cron
		status=$?
	elif [[ $lsb =~ "NeoKylin" || $PRETTY_NAME =~ "NeoKylin" ]]; then
		redhat_status_cron
		status=$?
	elif [[ $lsb =~ "Kylin" || $PRETTY_NAME =~ "Kylin" ]]; then
		ubuntu_status_cron
		status=$?
	fi
	return $status
}
# ============================== cron status check start ==============================

function stop_sfesp_func()
{
	local sfespconf=/sf/edr/agent/config/sfesp.conf
	if [ -f "${sfespconf}" ]; then
		sed -i "/^\[sfesp\]/,/^\[/ {/^\[sfesp\]/b;/^\[/b;s/^enable *=.*/enable=0/g;}" ${sfespconf}
		if [ $? -eq 0 ]; then
			return
		fi
	else
		sfespconf=/sangfor/edr/agent/config/sfesp.conf
		if [ -f "${sfespconf}" ]; then
			sed -i "/^\[sfesp\]/,/^\[/ {/^\[sfesp\]/b;/^\[/b;s/^enable *=.*/enable=0/g;}" ${sfespconf}
			if [ $? -eq 0 ]; then
				return
			fi
		else
			return
		fi
	fi

	chmod +x "${SCRIPT_PATH}/off_sfesp.sh"
	echo "* * * * * root ${SCRIPT_PATH}/off_sfesp.sh" > "${EDR_SFESP}"
	count=0
	while [ $count -lt 3 ]
	do
		echo -e "wait sfesp server stop ...."
		sleep 60
		chmod +x "${sfespconf}"
		if [ $? -eq 0 ]; then
			break
		fi
		((count++))
	done
	if [ -f "${EDR_SFESP}" ]; then
		rm -rf "${EDR_SFESP}"
	fi
	if [ $count -eq 3 ]; then
		echo -e "[Failed] The agent sfesp server stop failed"
		exit $ERR_ALREADY_INSTALL
	fi
	return
}

function system_env_check()
{
	echo -e "Start checking the installation requirements for the client."
	check_success=1
	# 1. check if os is domestic, if is domestic, exit
	checkIsGchOpSystem
	if [ $? -eq 0 ]; then
			echo -e "[Failed] The agent installation package is not compatible with the endpoint operating system."
			check_success=0
			exit 1
	fi

	if [ -f /etc/cron.d/edr_agent ]; then
		local content=`cat /etc/cron.d/edr_agent | grep "/sf/edr/agent/bin/eps_services_check.sh"`
		if [ -z "$content" ]; then
			content=`cat /etc/cron.d/edr_agent | grep "root /sangfor/edr/agent/bin/eps_services_check.sh"`
			if [ -z "$content" ]; then
				echo "[Failed] edr agent can not be installed with domestic."
				exit $ERR_ALREADY_INSTALL
			fi
		fi
	fi

	# 2. 'root' permission is required when installing.
	if [ `id -u` -ne 0 ];then
		echo -e "[Failed] The installer script requires root permissions."
		check_success=0
	fi

	# 3 full package install check
	if [ $full_package_install -ne 0 ];then
		# domain(or ip) and port are required when using full package installaion.
		if [ -z "$avai_ip" -o -z "$avai_port" ];then
			echo -e "[Failed] The IP and PORT are missing during full-package offline installation"
			check_success=0
		else
			checkport $avai_port
			if [ $? -ne 0 ]; then
				echo -e "[Failed] The port (${avai_port}) is invalid."
				check_success=0
			fi
			checkdest $avai_ip
			if [ $? -eq 0 ];then
				echo -e "[Failed] The IP address (${avai_ip}) is invalid."
				check_success=0
			fi
		fi
		# check offline components
		if [ ! -f $full_package ]; then
			echo -e "[Failed] Not found full-package file: $full_package."
			check_success=0
		fi
	else
		# 4. check port is valid
		get_port_info
		checkport $avai_port
		if [ $? -ne 0 ]; then
			echo -e "[Failed] The port (${avai_port}) is invalid."
			check_success=0
		else
			# 5. check ip is valid and check connet mgr
			get_available_mgr_addr
			if [ $? -ne 0 ]; then
				check_success=0
			fi
		fi
	fi

	# 6. check if have installed aES MGR
	if [ -f /etc/cron.d/eps_mgr -o -f /etc/cron.d/edr_mgr ]; then
		echo -e "[Failed] The agent cannot be installed on the server hosting the manager."
		check_success=0
	fi

	# 7. check cpu and system memory
	if [ $force_install_allowed == 0 ]; then 
		checkHardware
		hardware_check_result=$?
		if [[ $hardware_check_result -eq 2 ]] || [[ $hardware_check_result -eq 1 && $install_check -eq 1 ]];then
			#echo "your system have $cpuCores cpu and memory is $memtotalGB GB"
			if [ $install_check -eq 1 ]; then
					echo -e "[Failed] At least $CPU_RECOMMENDED_CORES CPU cores and $MEMORY_RECOMMENDED_GB GB of memory are required."
			else
					echo -e "[Failed] At least $CPU_REQUIREMENT_CORES CPU cores and $MEMORY_REQUIREMENT_GB GB of memory are required."
			fi
			check_success=0
		fi
	fi

	# 8. abort installaion when avalible disk space is less then 2.5G
	disk_must=`expr 2 \* 1024 \* 1024 \+ 512 \* 1024`
	avail_disk=`df / |grep / | sed -n '$p' |awk -F " " '{print $(NF-2)}'`
	if [ $avail_disk -lt $disk_must ]; then
		echo -e "[Failed] space of disk / is less than 2.5GB,installed failed."
		check_success=0
	fi

	# 9. check if can mkdir installdir
	default_dir=0
	if [ -z "$installdir" ]; then
		if [ ! -d /sf/edr/agent ];then
			default_dir=1
		fi
		installdir=/sf/edr/agent
	fi
	if [[ "${installdir:0:1}" != "/" ]]; then
		echo -e "[Failed] Install dir must be an absolute path: ${installdir}."
		check_success=0
	else
		if ! mkdir -p $installdir; then
			echo -e "[Failed] Failed to created a directory using the mkdir $installdir command."
			check_success=0
		fi
	fi

	# 10. check if install cron and start cron
	if [ ! -f /usr/sbin/cron ] && [ ! -f /usr/sbin/crond ]; then
		echo -e "[Failed] Unable to find Cron Service. The agent requires it to function properly."
		check_success=0
	else
		check_cron_status
		if [ $? -ne 0 ];then
			if [ -z "$enable_cron_cmd" ]; then
				echo -e "[Failed] Cron is not starting at boot. The agent requires it to function properly."
			else
				echo -e "[Failed] The cron service is not starting at boot. Please enable it by executing the \"$enable_cron_cmd\" command for the agent to function properly."
			fi
			check_success=0
		fi
	fi
	# 11. when install with proxy, check whether the proxy port is occupied
	if [ $proxy_install -eq 1 ]; then
		value=`${NET_TOOL} -tunlp|grep "${G2HPorxyIP}:${G2HProxyPort}"`
		if [ $? -eq 0 ];then
			echo $value | grep "${G2HProxyName}" >/dev/null 2>&1
			if [ $? -ne 0 ];then
				echo -e "[Failed] The G2HProxy port has been occupied."
				check_success=0
			fi
		fi
		avai_ip=$G2HPorxyIP
		avai_port=$G2HProxyPort
	fi

	# 12. check need tool
	get_need_tool
	check_need_tool
	if [ $? -ne 0 ]; then
		check_success=0
	fi

	# 13. check white system
	# if allowed force install , do not check system
	if [ $force_install_allowed == 0 ]; then 
		# check whether os_release and hardware  meet  minimum requierments
		checkOpSystemVersion

		osversion_check_result=$?
		if [[ $osversion_check_result -eq 2 ]] || [[ $osversion_check_result -eq 1 && $install_check -eq 1 ]];then
			# echo "your system is: $OS_NAME$OS_VERSION $OS_ARCHITECTURE"
			echo -e "[Failed] To ensure the best performance and security effect, only the following Operating Systems are supported:" 
			echo -e "         - Rocky: 8.4, 8.7, 8.8, 8.10, 9.2 \n         - Alma: 8.3, 8.9, 8.10, 9.2, 9.3, 9.4, 9.5 \n         - RedHat: 7.1-9.4 \n         - Debian: 9.1, 9.6, 9.7, 9.13, 11.9, 12.5 \n         - Ubuntu: 18.04, 20.04, 22.04, 24.04 \n         - SUSE: 15 \n         - CentOS: 7-8.5 \n         - Oracle: 8.x, 9.x \n         - Aliyun Linux: 2.1903, 3.2104"
			echo -e "         For more details, please refer to the user manual. Thank you for your understanding!" 
			check_success=0
		fi
	fi

	if [ $check_success -eq 0 ]; then
		exit 1
	fi
}


if [ -n "$1" ] && [ "$1" == "--help" ]; then
	return_usage
fi

# env check
system_env_check

stop_sfesp_func

echo "uid is $szuid."

if [ -z "$zonename" ];then
  if [ -f $ips_info ]; then
      zonename=`ReadINIfile "zone_id" "config" "$ips_info"`
      echo "read zone_name form ini ,zonename is_:$zonename ."
  fi
fi

# abort installaion when avalible mem is less then 500M
mem_must=`expr 500 \* 1024`
mem_free=`cat /proc/meminfo |grep -w MemFree |awk -F " " '{print $2}'`
mem_buff=`cat /proc/meminfo |grep -w Buffers |awk -F " " '{print $2}'`
mem_cache=`cat /proc/meminfo |grep -w Cached |awk -F " " '{print $2}'`
mem_free_total=`expr $mem_free \+ $mem_buff \+ $mem_cache`
if [ $mem_free_total -lt $mem_must ]; then
	echo "[Failed] system memory is less than 500MB,installed failed."
	exit $ERR_MEMORY_SHORTAGE
fi

# create /tmp if not exists
if [ ! -d "/tmp" ]; then
	mkdir -p /tmp
fi

function die()
{
	echo -e $*
	if [ $default_dir == 1 ];then
		rm -rf $installdir
	fi
	exit $2
}

if [ $system_check == 1 ]; then
    checkOpSystem
    if [ $? -ne 0 ]; then
        die "\nunsupport system" $ERR_UNSUPPORT_SYSTEM
    fi
fi

# remove the last uninstalling flag
if [ -f /sf/edr/agent/bin/uninstalling ]; then
	rm -rf /sf/edr/agent/bin/uninstalling
fi

# check if exit config
if [ ! -d "$installdir/config" ]; then
  # if not exist create it
  mkdir -p "$installdir/config"
fi

# backup agentid
if [ -f "/usr/share/sf/machineid" ]; then
	cp -f "/usr/share/sf/machineid" $installdir/config/
fi

# if default_dir_flag exists, when uninstalling, remove the install directory.
if [ $default_dir == 1 ];then
	touch $installdir/default_dir_flag
fi

function clean() {
	`rm -rf $SCRIPT_PATH/linux_*_installer* >/dev/null 2>&1`
	`rm -rf $SCRIPT_PATH/sfupdate* >/dev/null 2>&1`
	`rm -rf $SCRIPT_PATH/manager_info.txt >/dev/null 2>&1`
	`rm -rf $SCRIPT_PATH/off_sfesp.sh >/dev/null 2>&1`
	`rm -rf $SCRIPT_PATH/agent_installer.sh >/dev/null 2>&1`
	`rm -rf $SCRIPT_PATH/readme.txt >/dev/null 2>&1`
}

function full_package_unzip()
{
	local packages_path=$1
	local install_path=$2
	
	if [ ! -d "${install_path}" ];then
		mkdir -p ${install_path}
	else
		rm -rf ${install_path}
		mkdir -p ${install_path}
	fi
	
	tar -zxvf ${packages_path} -C ${install_path} >/dev/null 2>&1
}

longtype_length=`getconf LONG_BIT`
sfupdate="${SCRIPT_PATH}/sfupdate64.bin"
if [ $longtype_length -eq 32 ]; then
	sfupdate="${SCRIPT_PATH}/sfupdate32.bin"
fi
chmod +x $sfupdate
echo "start download edr module"
tempdir=${installdir##*sf}
if [[ $tempdir != "/edr/agent" && $tempdir != "/edr/agent/" ]];then
	lastChar=${installdir: -1}
	if [ $lastChar != "/" ];then
		installdir=$installdir"/sf/edr/agent"
	else
		installdir=$installdir"sf/edr/agent"
	fi
fi

mkdir -p $installdir/bin > /dev/null 2>&1
cp $sfupdate $installdir/bin/sfupdate
if [ $? -ne 0 ];then
	die "cp sfupdate fail."
fi

sfupdate="$installdir/bin/sfupdate"

INSTALL_RUNNING_FLAG=$installdir/bin/install_running_flag
touch ${INSTALL_RUNNING_FLAG}

# touch install flag(force install or ignore install check)
INSTALL_BYPASS_FLAG=$installdir/config/install_bypass_flag
if [ -f "$INSTALL_BYPASS_FLAG" ]; then
	rm -f $INSTALL_BYPASS_FLAG
fi
# delete and ten recreate install flag
if [ $force_install_allowed != 0 ] || [ $install_check != 1 ]; then
  touch $INSTALL_BYPASS_FLAG
fi

# save zone name to file group_info.ini
function check_zone_name_valid() {
	if [ -z $1 ];then
		return 1
	fi
	
	# Determine whether it consists of 32 characters, which are either digits or letters
	if ! [[ "$1" =~ ^[A-Za-z0-9]{32}([01])?$ ]]; then
		return 1
	fi
	return 0
}

echo "zonename=$zonename"
if [ "$zonename" != "" ];then
	check_zone_name_valid $zonename
	if [ $? -ne 0 ]; then
		echo "Error: invalid zone_name: $zonename"
		zonename=""
	else
		echo -e "[agent_group_info]\nzoneId=$zonename" >> "$installdir/config/group_info.ini"
	fi
fi

if [ "$szuid" = "" ]; then
	echo "curr install path: $installdir url:https://${avai_ip}:${avai_port}"
	if [ "$full_package_install" -ne 0 ];then
		# full package installation
		full_package_unzip $full_package $installdir/packages
		$sfupdate -a "https://${avai_ip}:${avai_port}" -d $installdir -b 0 -w
	else
		$sfupdate -a "https://${avai_ip}:${avai_port}" -d $installdir -c -b 0
	fi
	
else
	echo "curr install path: $installdir url:https://${avai_ip}:${avai_port} uid :$szuid"
	if [ $full_package_install -ne 0 ];then
		# full package installation
		full_package_unzip $full_package $installdir/packages
		$sfupdate -a "https://${avai_ip}:${avai_port}" -d $installdir -u $szuid -b 0 -w
	else
		$sfupdate -a "https://${avai_ip}:${avai_port}" -d $installdir -c -u $szuid -b 0
	fi
fi

exit_code=$?
if [ $exit_code -ne 0 ];then
	rm -f ${INSTALL_RUNNING_FLAG}
	die "[Failed] Failed to download the agent installation package." $exit_code
fi

SERVICE_NAME="eps_services"
EpsServicesPath=$installdir/bin/$SERVICE_NAME

if [ -f "${EpsServicesPath}" ];then
    ${EpsServicesPath} restart
fi

# clean files when slient installing is ended.
if [ $slient_install -eq 1 ]; then
	clean
fi

rm -f ${INSTALL_RUNNING_FLAG}
echo "Agent installation successful."
